-- A scripted implementation of a PBR material frontend in the Specular/Gloss mode
-- Material_PBRSpecGloss.ms, by Zap Andersson and Juan Martinez
-- Modified: 2020-01-15
-- Copyright 2020 Autodesk Inc, All rights reserved. This file is licensed under Apache 2.0 license
-- You are free to modify this file to make derivative versions to match your workflow.
-- Remember to change the ClassID if you do...
--    https://github.com/ADN-DevTech/3dsMax-OSL-Shaders/blob/master/LICENSE.txt

plugin material PBRSpecGloss
name:~PBR_MATERIAL_SPGL~
classID:#(0xd00f1e54, 0x01dbad33)
extends:PhysicalMaterial replaceUI:true version:2019.1215
(
	local params
	local settings
	
	fn setGamma1 map = 
	(
		if (classof map == bitmapTex) do
		(
			if (map.bitmap != undefined) do 
			(
				if (map.bitmap.gamma != 1.0) do 
				(
					local gamma1bmp = openBitmap map.bitmap.filename gamma:1.0
					if (gamma1bmp != undefined) do map.bitmap = gamma1bmp
				)					
			)				
		)
		if (classOf map == OSLMap) do 
		(
			if (hasProperty map "autoGamma") do (
				map.autogamma = false
				
				if (hasProperty map "manualGamma") do 
					map.manualGamma = 1.0
			)
		)
		if (classOf map == MultiOutputChannelTexmapToTexmap) do
		(
			setGamma1(map.sourceMap)
		)
	)
	
	fn transmap = (
		-- Is the map there, and already connected to the OSL map - use it
		if (delegate.roughness_map != undefined) do
		(
			if (delegate.roughness_map.sourceMap != undefined) do
				return delegate.roughness_map.sourceMap
		)
		-- If not - create it

		local code = 
		"shader SpecGlossToPhysical
		(
			color        In     = 0.02,
			float        Gloss  = 1.0,
			float        AO     = 1.0,
			int          Invert = 1,
			output float Reflectivity = 0.0,
			output color ReflColor = 0.0,
			output float Roughness = 0.0
		)
		{
			Roughness = Invert?1.0 - Gloss:Gloss;
	
			Reflectivity = max(max(In[0],In[1]),In[2]);
			if (Reflectivity > 0.0)
				ReflColor = In / Reflectivity;

			float fres   = pow(1-abs(dot(I, N)),5.0) * (1.0-Roughness);
			Reflectivity = mix(Reflectivity, 1.0, fres);
			ReflColor    = mix(ReflColor, 1.0, fres) * AO;
		}"

		local tm = OSLmap()
		tm.OSLCode = code			
		
		local mul = MultiOutputChannelTexmapToTexmap()
		
		mul.sourceMap = tm
		mul.outputChannelIndex = 1
		delegate.reflectivity_map = mul

		mul = MultiOutputChannelTexmapToTexmap()
		mul.sourceMap = tm
		mul.outputChannelIndex = 2
		delegate.refl_color_map = mul

		mul = MultiOutputChannelTexmapToTexmap()
		mul.sourceMap = tm
		mul.outputChannelIndex = 3
		delegate.roughness_map = mul
		return tm
	)

	parameters settings rollout:settings
	(
		ao_affects_diffuse     type:#boolean ui:aoDiffuse default:true
		ao_affects_reflection type:#boolean ui:aoReflect default:true
		normal_flip_red         type:#boolean ui:normFlipR default:false
		normal_flip_green      type:#boolean ui:normFlipG default:false
	)
	
	
    parameters main rollout:params 
    (
        basecolor type:#frgba default:white ui:cpCol 
        base_color_map type:#TextureMap ui:mapBaseColor 
        specular type:#frgba default:[0.04,0.04,0.04,1.0] ui:cpSpecular
        specular_map type:#TextureMap ui:mapSpecular
        glossiness type:#float default:0.0 ui:spnGlossiness
        glossiness_map type:#TextureMap ui:mapGlossiness
        useGlossiness type:#integer default:1
        ao_map type:#TextureMap ui:mapAmbientOcclusion
        bump_map_amt type:#float default:1.0 ui:spnBumpAmount
        norm_map type:#TextureMap ui:mapNormal
        emit_color type:#frgba default:black ui:cpEmCol
        emit_color_map type:#TextureMap ui:mapEmitColor
		opacity_map type:#TextureMap ui:mapOpacity
		
        on basecolor set val do (
			delegate.base_color = val
			
			if (basecolor != white) then
			(
				local mul = OSLMap()
				mul.oslCode = "shader mul (color a = 1, color b = 1, output color out = 1) { out = a * b; }"
				mul.a = basecolor
				mul.b_map = base_color_map
				delegate.base_color_map = mul
			)
			else
			(
				delegate.base_color_map = base_color_map
			)
		)
        on base_color_map set val do (
			if (basecolor != white) then
			(
				local mul = OSLMap()
				mul.oslCode = "shader mul (color a = 1, color b = 1, output color out = 1) { out = a * b; }"
				mul.a = basecolor
				mul.b_map = base_color_map
				delegate.base_color_map = mul
			)
			else
			(
				delegate.base_color_map = base_color_map
			)
		)
        on specular set val do (
			local x = transmap()
			x.In = val
		)
        on specular_map set val do (
			local x = transmap()
			x.In_map = val
		)
        on glossiness set val do 
		(
			local x = transmap()
			x.Gloss = val
		)
        on glossiness_map set val do (
			setGamma1(val)
			local x = transmap()
			x.Gloss_map = val
		)
		
        on useGlossiness set val do (
			local x = transmap()
			x.Invert = (val == 1)
		)
		
        on ao_map set val do (
			setGamma1(val)
			
			if (ao_affects_diffuse) then
				delegate.base_weight_map = val		
			else
				delegate.base_weight_map = undefined
			
			local x = transmap()
			
			if (ao_affects_reflection) then
				x.AO_map = val
			else
				x.AO_map = undefined
		)
        on opacity_map set val do (
			setGamma1(val)
			delegate.cutout_map = val
		)
        on bump_map_amt set val do delegate.bump_map_amt = val
        on norm_map set val do (
			if (val != undefined) then (
				setGamma1(val)
				local bm = Gnormal()
				bm.normal_map = val
				bm.flipred    = normal_flip_red
				bm.flipgreen = normal_flip_green
				delegate.bump_map = bm
			)
			else
			(
				delegate.bump_map = undefined
			)
		)
        on emit_color set val do delegate.emit_color = val
        on emit_color_map set val do delegate.emit_color_map = val
    )
	
    rollout params ~PBR_MATERIAL_PARAMSPEC~
    (
		edittext lvl readOnly:true text:~PBR_MATERIAL_INFO~ height:60
		group ~PBR_MATERIAL_MTLPARS~ (	
			colorpicker cpCol ~PBR_MATERIAL_BASECOL~ color:white across:2 fieldWidth:45 offset:[-45,6] align:#right
			mapbutton mapBaseColor "" align:#right width:190 offset:[-5,5]

			colorpicker cpSpecular ~PBR_MATERIAL_SPECULAR~ color:white across:2 fieldWidth:45 offset:[-45,3] align:#right
			mapbutton mapSpecular "" align:#right width:190 offset:[-5,2]

			spinner spnGlossiness ~PBR_MATERIAL_GLOSSINESS~ range:[0.0,1.0,0.0] scale:0.01 across:2 fieldWidth:35 offset:[-45,5] align:#right
			mapbutton mapGlossiness "" align:#right width:190 offset:[-5,2]

			label lblAO ~PBR_MATERIAL_AO~ across:2 align:#right offset:[-45,5]
			mapbutton mapAmbientOcclusion "" align:#right width:190 offset:[-5,2]

			spinner spnBumpAmount ~PBR_MATERIAL_NORMAL~ range:[-5.0,5.0,1.0] scale:0.01 fieldwidth:35 across:2 offset:[-45,5] align:#right
			mapbutton mapNormal "" align:#right width:190 offset:[-5,2]

			colorpicker cpEmCol ~PBR_MATERIAL_EMISSION~ color:black across:2 fieldWidth:45 offset:[-45,3] align:#right
			mapbutton mapEmitColor "" align:#right width:190 offset:[-5,2]

			label lblOpac ~PBR_MATERIAL_OPACITY~ across:2 across:2 align:#right offset:[-45,5] height:28
			mapbutton mapOpacity "" align:#right width:190 offset:[-5,2]
			
		)
    )
	
	rollout settings ~PBR_MATERIAL_SETTINGS~
	(
		label lblSurface ~PBR_MATERIAL_SURFSMOOTH~ across:2 align:#left
		radiobuttons rdoGlossiness labels:#(~PBR_MATERIAL_GLOSSINESS2~, ~PBR_MATERIAL_ROUGHNESS2~) default:1 align:#left
		checkbox aoDiffuse ~PBR_MATERIAL_AO_AFFECTS_DIFFUSE~ offset:[20,0] across:2
		checkBox normFlipR ~PBR_MATERIAL_FLIP_R~ align:#left 
		checkbox aoReflect ~PBR_MATERIAL_AO_AFFECTS_REFLECT~ offset:[20,0] across:2
		checkBox normFlipG ~PBR_MATERIAL_FLIP_G~ align:#left 		
			label lblSpc "" height:2	
		
        fn setUIText =
        (
            if useGlossiness == 1 then
            (
                params.spnGlossiness.caption = ~PBR_MATERIAL_GLOSSINESS~
            )
            else
            (
                params.spnGlossiness.caption = ~PBR_MATERIAL_ROUGHNESS~
            )
        )
		
        on settings open do
        (
            rdoGlossiness.state = useGlossiness
            setUIText()
        )

        on rdoGlossiness changed state do
        (
            useGlossiness = state
            setUIText()
        )
		
		on aoDiffuse changed state do ao_map = ao_map
		on aoReflect changed state do ao_map = ao_map
		on normFlipR changed state do norm_map = norm_map			
		on normFlipG changed state do norm_map = norm_map			
	)

    on create do
    (
		-- sorry, gamma off will not work with this material 
		if iDisplayGamma.colorCorrectionMode == #none do (
			-- for those stuck in the literal dark ages - turn it ON
			iDisplayGamma.colorCorrectionMode = #gamma
			iDisplayGamma.gamma= 2.2
			iDisplayGamma.affectColorPickers=true
			iDisplayGamma.affectMEdit=true
		)
		
		-- Switch to Advanced material mode
		local vps = NitrousGraphicsManager.GetActiveViewportSetting()
		vps.ShowMaterialOption=#realistic
		-- Switch to "High Quality" mode
		--actionMan.executeAction -844228238 "12"

        delegate.base_color = white
        delegate.metalness = 0.0
		delegate.trans_ior = 50.0
        delegate.roughness = 0.0
        delegate.bump_map_amt = 1.0
        delegate.roughness_inv = false
        delegate.reflectivity = 1.0
        delegate.emit_color = black
        delegate.emit_color_map_on = true
    )
)
-------BEGIN-SIGNATURE-----
-- 4wYAADCCBt8GCSqGSIb3DQEHAqCCBtAwggbMAgEBMQ8wDQYJKoZIhvcNAQELBQAw
-- CwYJKoZIhvcNAQcBoIIE3jCCBNowggPCoAMCAQICEDUAFkMQxqI9PltZ2eUG16Ew
-- DQYJKoZIhvcNAQELBQAwgYQxCzAJBgNVBAYTAlVTMR0wGwYDVQQKExRTeW1hbnRl
-- YyBDb3Jwb3JhdGlvbjEfMB0GA1UECxMWU3ltYW50ZWMgVHJ1c3QgTmV0d29yazE1
-- MDMGA1UEAxMsU3ltYW50ZWMgQ2xhc3MgMyBTSEEyNTYgQ29kZSBTaWduaW5nIENB
-- IC0gRzIwHhcNMTkwNjI1MDAwMDAwWhcNMjAwODA3MjM1OTU5WjCBijELMAkGA1UE
-- BhMCVVMxEzARBgNVBAgMCkNhbGlmb3JuaWExEzARBgNVBAcMClNhbiBSYWZhZWwx
-- FzAVBgNVBAoMDkF1dG9kZXNrLCBJbmMuMR8wHQYDVQQLDBZEZXNpZ24gU29sdXRp
-- b25zIEdyb3VwMRcwFQYDVQQDDA5BdXRvZGVzaywgSW5jLjCCASIwDQYJKoZIhvcN
-- AQEBBQADggEPADCCAQoCggEBAMsptjSEm+HPve6+DClr+K4CgrtrONjtHxHBwTMC
-- mrwF9bnsdMiSgvYigTKk858TlqVs7GiBVLD3SaSZqfSXOv7L55i965L+wIx0EZxX
-- xDzbyLh1rLSSNWO8oTDIKnPsiwo5x7CHRUi/eAICOvLmz7Rzi+becd1j/JPNWe5t
-- vum0GL/8G4vYICrhCycizGIuv3QFqv0YPM75Pd2NP0V4W87XPeTrj+qQoRKMztJ4
-- WNDgLgT4LbMBIZyluU8iwXNyWQ8FC2ya3iJyy0EhZhAB2H7oMrAcV1VJJqwZcZQU
-- XMJTD+tuCqKqJ1ftv1f0JVW2AADnHgvaB6E6Y9yR/jnn4zECAwEAAaOCAT4wggE6
-- MAkGA1UdEwQCMAAwDgYDVR0PAQH/BAQDAgeAMBMGA1UdJQQMMAoGCCsGAQUFBwMD
-- MGEGA1UdIARaMFgwVgYGZ4EMAQQBMEwwIwYIKwYBBQUHAgEWF2h0dHBzOi8vZC5z
-- eW1jYi5jb20vY3BzMCUGCCsGAQUFBwICMBkMF2h0dHBzOi8vZC5zeW1jYi5jb20v
-- cnBhMB8GA1UdIwQYMBaAFNTABiJJ6zlL3ZPiXKG4R3YJcgNYMCsGA1UdHwQkMCIw
-- IKAeoByGGmh0dHA6Ly9yYi5zeW1jYi5jb20vcmIuY3JsMFcGCCsGAQUFBwEBBEsw
-- STAfBggrBgEFBQcwAYYTaHR0cDovL3JiLnN5bWNkLmNvbTAmBggrBgEFBQcwAoYa
-- aHR0cDovL3JiLnN5bWNiLmNvbS9yYi5jcnQwDQYJKoZIhvcNAQELBQADggEBADo7
-- 6cASiVbzkjsADk5MsC3++cj9EjWeiuq+zzKbe55p6jBNphsqLUvMw+Z9r2MpxTEs
-- c//MNUXidFsslWvWAUeOdtytNfhdyXfENX3baBPWHhW1zvbOPHQLyz8LmR1bNe9f
-- R1SLAezJaGzeuaY/Cog32Jh4qDyLSzx87tRUJI2Ro5BLA5+ELiY21SDZ7CP9ptbU
-- CDROdHY5jk/WeNh+3gLHeikJSM9/FPszQwVc9mjbVEW0PSl1cCLYEXu4T0o09ejX
-- NaQPg10POH7FequNcKw50L63feYRStDf6GlO4kNXKFHIy+LPdLaSdCQL2/oi3edV
-- MdpL4F7yw1zQBzShYMoxggHFMIIBwQIBATCBmTCBhDELMAkGA1UEBhMCVVMxHTAb
-- BgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8wHQYDVQQLExZTeW1hbnRlYyBU
-- cnVzdCBOZXR3b3JrMTUwMwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1NiBD
-- b2RlIFNpZ25pbmcgQ0EgLSBHMgIQNQAWQxDGoj0+W1nZ5QbXoTANBgkqhkiG9w0B
-- AQsFADANBgkqhkiG9w0BAQEFAASCAQB/aYZyPZeZBi1BA7qvxp2v5cVPq8gqqxKF
-- dRKuprdr30ixeVpXM1wrHYqRqnlR2RRxwGFCK2IyomiYOMi5cr4yv2MbuDHWG6c1
-- Xk19fxiWMLLj7agSXbeeIJqwBxi5o9jdqQ/kHUgljRdJfwbnkBrzcRsJ1hB5ZNi8
-- ew8F2J8TFEfL7c4jD65DXJmb3Dhm3TCPCDuYB37lJc34ZAk5Uf/GTfHcJo03OyoW
-- 9wTSoiV/GFz5lXfpVsBioZS4j1Pn3L0IoiO5cpJCGmRunabQ/oVWaVG0Yd/KKP4z
-- auVgY59NCUDR6ALLlwxibpyFahKvpJopS5mgwOQ6qvvdYRa7qHpo
-- -----END-SIGNATURE-----